function [bestc, bestp, CV] = ottimiz_param_SVM_LOSO(inst_sc, labels, log2c, log2p, sogg_label, plotflag, scalingtype)
% SVM RBF kernel parameter optimization (C and Gamma) obtained from 
% parameters grid search. The optimization criterion is defined on 
% leave-one-subject-out (LOSO) cross-validation results
% 
% [bestc, bestp] = ottimiz_param_SVM(inst_sc, labels, log2c, log2p, nfold, plotflag)
% 
% inst_sc     = feature set (prescaled/normalized/standardized)
% labels      = classification labels
% log2c       = C parameter values expressed as log2(C)
% log2p       = gamma parameter values expressed as log2(gamma)
% sogg_label  = subject-label for each instance (needed for LOSO cross validation)
% plotflag    = graphical output is displayed if this flag is set to true
% scalingtype = set to 1 for normalization, set to 2 for standardization
% 
% example
% [bestc, bestp] = ottimiz_param_SVM(inst_sc,labels,[0:4:16],[10:-5:-5],sogg_label,1,1)
% 
% [Andrea Mannini: a.mannini@sssup.it; Last modified: Nov 23 2010]


% log2c = [0:4:16];
% log2p = [10:-5:-5];

Lc = length(log2c);
Lp = length(log2p); 
CV = zeros(Lc,Lp);
l2C = zeros(Lc,Lp);
l2P = zeros(Lc,Lp);

soggetti= unique(sogg_label);
S = length(soggetti);
tic;
h = waitbar( 0,'Grid search: ottimizzazione parametri');

for c = 1:Lc
    for g = 1:Lp
        cv2 = 0;
        for i = 1:S
            sogg = soggetti(i);
            
            Inst_tr =  inst_sc(sogg_label~=sogg,:);
            
            
            switch scalingtype
                case 1
                    sc_low = -1; sc_up = 1;
                    [Inst_tr_n, M, m] = scaling_SVM(Inst_tr,sc_low,sc_up);
                    
                case 2
                    [Inst_tr_n, mu, sigma] = featureNormalize(Inst_tr); % calcolo mu e sigma dal training set
            end
            
            [m_tr,n2] = size(Inst_tr_n);
            Inst_tr_n = [ones(m_tr, 1) Inst_tr_n];

            cmd = [' -s 0 -t 2 -c ', num2str(2^log2c(c), '%12.12f'), ' -g ', num2str(2^log2p(g), '%12.12f')];
            model = svmtrain(labels(sogg_label~=sogg), Inst_tr_n, cmd);
            
            Inst_te = inst_sc(sogg_label==sogg,:);
            Inst_te_n = Inst_te;
            switch scalingtype
                case 1
                    for e = 1:size(Inst_te,2)
                        Inst_te_n(:,e) = ones(length(Inst_te(:,e)),1)*sc_low + (Inst_te(:,e)-ones(length(Inst_te(:,e)),1)*m(e)) .* (sc_up-sc_low)/(M(e)-m(e));
                    end
                case 2
                    Inst_te_n = bsxfun(@minus, Inst_te_n, mu);
                    Inst_te_n = bsxfun(@rdivide, Inst_te_n, sigma);
            end
            
            [m2_te,n2] = size(Inst_te_n);
            Inst_te_n = [ones(m2_te, 1) Inst_te_n];
                        

            
%             [output, cv(i,:)] = svmpredict(labels(sogg_label==sogg), Inst_te_n, model);
            [output, cv(i,:)] = svmpredict(labels(sogg_label==sogg), Inst_te_n, model);
        
            cv2 = cv2+(cv(i,1)*sum(sogg_label==sogg));
        end
        
        
%         CV(c,g) = mean(cv(:,2));
        CV(c,g) = cv2 / length(labels);
        l2C(c,g) = log2c(c);
        l2P(c,g) = log2p(g);
        waitbar( ((c-1)*Lp + g)/(Lc*Lp) ,h);
    end
%     waitbar((c)/(Lc),h);
end
close(h);
toc;

[bestcv,ind_best] = max(CV(:));
[ind_bestlc, ind_bestlp] = ind2sub(size(CV),ind_best);

bestlc1 = log2c(ind_bestlc)
bestlp1 = log2p(ind_bestlp)
bestc = 2^ bestlc1;
bestp = 2^ bestlp1;



if (plotflag)
%%% Figura
figure1 = figure('Units', 'normalized', 'Position',[0.2 0.1 0.6 0.8]);
axes1 = axes('Parent',figure1,'Position',[0.1 0.2314 0.7446 0.621],...
    'Layer','top');
box('on'); grid('off');

output1.l2C = l2C;
output1.l2P = l2P;
output1.CV = CV;
output1.bestc = bestc;
output1.bestp = bestp;
output1.bestcv = bestcv;
assignin('base','outputopt',output1);

[cc,h] = contour(l2C,l2P,CV,linspace(min(CV(:)),max(CV(:)),50));
% [cc,h] = contourf(l2C,l2P,CV,linspace(min(CV(:)),max(CV(:)),50));
% [cc,h] = contourf(l2C,l2P,CV, [0:.1:105]);
h = clabel(cc,h,linspace(min(CV(:)), max(CV(:)), 50) );
% h = clabel(cc,h, [0:5:105]);
set(h,'BackgroundColor',[1 1 .6])
xlabel('log_2 (C)');
ylabel('log_2 (\gamma)');
title(['Support Vector Classification: optimization of parameters' ], 'FontWeight', 'Bold');
% Create colorbar
colorbar('peer',axes1,'ZGrid','on','YGrid','on','XGrid','on');
% Resize the axes in order to prevent it from shrinking.
set(axes1,'Position',[0.1 0.2314 0.7446 0.621]);
% Create textbox
annotation(figure1,'textbox',[0.01347 0.01514 0.9633 0.1038],...
    'String',{['Best parameters set: log_2(C) = ' num2str(log2(bestc)) ', log_2(\gamma)= ' num2str(log2(bestp))];...
    ['C = ' num2str(bestc) ', \gamma= ' num2str(bestp) ', \bfMSE = ' num2str(bestcv) '']},...
    'FitBoxToText','off',...
    'BackgroundColor',[1 1 1]);
end
